/**
 * Something that is either an actual Error, or something very similar,
 * on which we can start examining error properties.
 */
export type ErrorLike = Partial<Error> & {
    code?: string;
    cmd?: string;
    signal?: string;
    statusCode?: number;
    statusMessage?: string;
    cause?: ErrorLike | Error | unknown;
};
/**
 * An easy way to check if an unknown object (e.g. a catch(e) argument) is
 * actually an error-like
 */
export declare function isErrorLike(error: any): error is ErrorLike;
/**
 * A convenience method to make something error-ish into an actual Error instance.
 */
export declare function asErrorLike(error: any): ErrorLike;
export declare class CustomError extends Error implements ErrorLike {
    constructor(message?: string, extras?: {
        code?: string;
        statusCode?: number;
        cause?: Error;
    });
    readonly code?: string;
    readonly statusCode?: number;
    readonly cause?: Error;
}
export declare class StatusError extends CustomError {
    constructor(
    /**
     * Should be a valid HTTP status code
     */
    statusCode: number, message: string, extras?: {
        code?: string;
        cause?: Error;
    });
}
/**
 * An error to throw in expected-never cases - by using this, you ask TypeScript to
 * be sure that it agrees that the case is truly unreachable.
 */
export declare class UnreachableCheck extends CustomError {
    constructor(value: never, getValue?: (v: any) => any);
}
/**
 * A convenient expression-shaped version of UnreachableCheck
 */
export declare const unreachableCheck: (value: never, getValue?: (v: any) => any) => never;
