import * as cors from 'cors';
import { AdminPlugin, PluginConstructorMap, PluginStartParamsMap } from './admin-plugin-types';
export interface AdminServerOptions<Plugins extends {
    [key: string]: AdminPlugin<any, any>;
}> {
    /**
     * Should the admin server print extra debug information? This enables admin server debugging
     * only - individual mock session debugging must be enabled separately.
     */
    debug?: boolean;
    /**
     * Set CORS options to limit the sites which can send requests to manage this admin server.
     */
    corsOptions?: cors.CorsOptions & {
        strict?: boolean;
        allowPrivateNetworkAccess?: boolean;
    };
    /**
     * Set a keep alive frequency in milliseconds for the subscription & stream websockets of each
     * session, to ensure they remain connected in long-lived connections, especially in browsers which
     * often close quiet background connections automatically.
     */
    webSocketKeepAlive?: number;
    /**
     * Override the default parameters for sessions started from this admin server. These values will be
     * used for each setting that is not explicitly specified by the client when creating a mock session.
     */
    pluginDefaults?: Partial<PluginStartParamsMap<Plugins>>;
    /**
     * Some rule options can't easily be specified in remote clients, since they need to access
     * server-side state or Node APIs directly. To handle this, referenceable parameters can
     * be provided here, and referenced with a `{ [MOCKTTP_PARAM_REF]: <value> }` value in place
     * of the real parameter in the remote client.
     */
    ruleParameters?: {
        [key: string]: any;
    };
    /**
     * @internal
     *
     * This API is not yet stable, and is intended for internal use only. It may change in future
     * in minor versions without warning.
     *
     * This defines admin plugin modules: remote-controlled types of mocks that should be attached to this
     * admin server, to allow configuring other mocking services through the same HTTP infrastructure.
     *
     * This can be useful when mocking non-HTTP protocols like WebRTC.
     */
    adminPlugins?: PluginConstructorMap<Plugins>;
}
export declare class AdminServer<Plugins extends {
    [key: string]: AdminPlugin<any, any>;
}> {
    private debug;
    private requiredOrigin;
    private webSocketKeepAlive;
    private ruleParams;
    private app;
    private server;
    private eventEmitter;
    private adminPlugins;
    private sessions;
    constructor(options?: AdminServerOptions<Plugins>);
    resetAdminServer(): Promise<void>;
    /**
     * Subscribe to hear when each mock ession is started. The listener is provided the
     * session plugin data, which can be used to log session startup, add side-effects that
     * run elsewhere at startup, or preconfigure every started plugin in addition ways.
     *
     * This is run synchronously when a session is created, after it has fully started
     * but before its been returned to remote clients.
     */
    on(event: 'mock-session-started', listener: (plugins: Plugins, sessionId: string) => void): void;
    /**
     * Subscribe to hear when a mock session is stopped. The listener is provided with
     * the state of all plugins that are about to be stopped. This can be used to log
     * mock session shutdown, add side-effects that run elsewhere at shutdown, or clean
     * up after sessions in other ways.
     *
     * This is run synchronously immediately before the session is shutdown, whilst all
     * its state is still available, and before remote clients have had any response to
     * their request. This is also run before shutdown when the admin server itself is
     * cleanly shutdown with `adminServer.stop()`.
     */
    on(event: 'mock-session-stopping', listener: (plugins: Plugins, sessionId: string) => void): void;
    start(listenOptions?: number | {
        port: number;
        host: string;
    }): Promise<void>;
    private startSessionManagementAPI;
    stop(): Promise<void>;
    private static baseSchema;
    private buildBaseResolvers;
    private resetPluginsForSession;
    private enableDebugForSession;
    get ruleParameterKeys(): string[];
}
//# sourceMappingURL=admin-server.d.ts.map