import { CompletedRequest, Method, Explainable, OngoingRequest } from "../types";
import { Serializable, ClientServerChannel } from "../serialization/serialization";
import { MaybePromise } from '../util/type-utils';
export interface RequestMatcher extends Explainable, Serializable {
    type: keyof typeof MatcherLookup;
    matches(request: OngoingRequest): MaybePromise<boolean>;
}
export interface SerializedCallbackMatcherData {
    type: string;
    name?: string;
    version?: number;
}
export declare class WildcardMatcher extends Serializable implements RequestMatcher {
    readonly type = "wildcard";
    matches(): boolean;
    explain(): string;
}
export declare class MethodMatcher extends Serializable implements RequestMatcher {
    method: Method;
    readonly type = "method";
    constructor(method: Method);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class ProtocolMatcher extends Serializable implements RequestMatcher {
    protocol: "http" | "https" | "ws" | "wss";
    readonly type = "protocol";
    constructor(protocol: "http" | "https" | "ws" | "wss");
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class HostMatcher extends Serializable implements RequestMatcher {
    host: string;
    readonly type = "host";
    constructor(host: string);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class HostnameMatcher extends Serializable implements RequestMatcher {
    readonly hostname: string;
    readonly type = "hostname";
    constructor(hostname: string);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class PortMatcher extends Serializable implements RequestMatcher {
    readonly type = "port";
    port: string;
    constructor(port: number);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class SimplePathMatcher extends Serializable implements RequestMatcher {
    path: string;
    readonly type = "simple-path";
    constructor(path: string);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class RegexPathMatcher extends Serializable implements RequestMatcher {
    readonly type = "regex-path";
    readonly regexSource: string;
    readonly regexFlags: string;
    constructor(regex: RegExp);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class RegexUrlMatcher extends Serializable implements RequestMatcher {
    readonly type = "regex-url";
    readonly regexSource: string;
    readonly regexFlags: string;
    constructor(regex: RegExp);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class HeaderMatcher extends Serializable implements RequestMatcher {
    readonly type = "header";
    headers: {
        [key: string]: string;
    };
    constructor(headersInput: {
        [key: string]: string;
    });
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class ExactQueryMatcher extends Serializable implements RequestMatcher {
    query: string;
    readonly type = "exact-query-string";
    constructor(query: string);
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class QueryMatcher extends Serializable implements RequestMatcher {
    readonly type = "query";
    queryObject: {
        [key: string]: string | string[];
    };
    constructor(queryObjectInput: {
        [key: string]: string | number | (string | number)[];
    });
    matches(request: OngoingRequest): boolean;
    explain(): string;
}
export declare class FormDataMatcher extends Serializable implements RequestMatcher {
    formData: {
        [key: string]: string;
    };
    readonly type = "form-data";
    constructor(formData: {
        [key: string]: string;
    });
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare type MultipartFieldMatchCondition = {
    name?: string;
    filename?: string;
    content?: string | Uint8Array;
};
export declare class MultipartFormDataMatcher extends Serializable implements RequestMatcher {
    matchConditions: Array<MultipartFieldMatchCondition>;
    readonly type = "multipart-form-data";
    constructor(matchConditions: Array<MultipartFieldMatchCondition>);
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class RawBodyMatcher extends Serializable implements RequestMatcher {
    content: string;
    readonly type = "raw-body";
    constructor(content: string);
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class RawBodyIncludesMatcher extends Serializable implements RequestMatcher {
    content: string;
    readonly type = "raw-body-includes";
    constructor(content: string);
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class RegexBodyMatcher extends Serializable implements RequestMatcher {
    readonly type = "raw-body-regexp";
    readonly regexString: string;
    constructor(regex: RegExp);
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class JsonBodyMatcher extends Serializable implements RequestMatcher {
    body: {};
    readonly type = "json-body";
    constructor(body: {});
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class JsonBodyFlexibleMatcher extends Serializable implements RequestMatcher {
    body: {};
    readonly type = "json-body-matching";
    constructor(body: {});
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class CookieMatcher extends Serializable implements RequestMatcher {
    cookie: {
        [key: string]: string;
    };
    readonly type = "cookie";
    constructor(cookie: {
        [key: string]: string;
    });
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
}
export declare class CallbackMatcher extends Serializable implements RequestMatcher {
    callback: (request: CompletedRequest) => MaybePromise<boolean>;
    readonly type = "callback";
    constructor(callback: (request: CompletedRequest) => MaybePromise<boolean>);
    matches(request: OngoingRequest): Promise<boolean>;
    explain(): string;
    /**
     * @internal
     */
    serialize(channel: ClientServerChannel): SerializedCallbackMatcherData;
    /**
     * @internal
     */
    static deserialize({ name }: SerializedCallbackMatcherData, channel: ClientServerChannel): CallbackMatcher;
}
export declare const MatcherLookup: {
    wildcard: typeof WildcardMatcher;
    method: typeof MethodMatcher;
    protocol: typeof ProtocolMatcher;
    host: typeof HostMatcher;
    hostname: typeof HostnameMatcher;
    port: typeof PortMatcher;
    'simple-path': typeof SimplePathMatcher;
    'regex-path': typeof RegexPathMatcher;
    'regex-url': typeof RegexUrlMatcher;
    header: typeof HeaderMatcher;
    query: typeof QueryMatcher;
    'exact-query-string': typeof ExactQueryMatcher;
    'form-data': typeof FormDataMatcher;
    'multipart-form-data': typeof MultipartFormDataMatcher;
    'raw-body': typeof RawBodyMatcher;
    'raw-body-regexp': typeof RegexBodyMatcher;
    'raw-body-includes': typeof RawBodyIncludesMatcher;
    'json-body': typeof JsonBodyMatcher;
    'json-body-matching': typeof JsonBodyFlexibleMatcher;
    cookie: typeof CookieMatcher;
    callback: typeof CallbackMatcher;
};
export declare function matchesAll(req: OngoingRequest, matchers: RequestMatcher[]): Promise<boolean>;
export declare function explainMatchers(matchers: RequestMatcher[]): string;
//# sourceMappingURL=matchers.d.ts.map