"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebSocketRule = void 0;
const uuid_1 = require("uuid");
const types_1 = require("../../types");
const request_utils_1 = require("../../util/request-utils");
const rule_serialization_1 = require("../rule-serialization");
const matchers = require("../matchers");
const websocket_handlers_1 = require("./websocket-handlers");
class WebSocketRule {
    constructor(data) {
        this.requests = [];
        this.requestCount = 0;
        (0, rule_serialization_1.validateMockRuleData)(data);
        this.id = data.id || (0, uuid_1.v4)();
        this.priority = data.priority ?? types_1.RulePriority.DEFAULT;
        this.matchers = data.matchers;
        if ('handle' in data.handler) {
            this.handler = data.handler;
        }
        else {
            // We transform the definition into a real handler, by creating an raw instance of the handler (which is
            // a subtype of the definition with the same constructor) and copying the fields across.
            this.handler = Object.assign(Object.create(websocket_handlers_1.WsHandlerLookup[data.handler.type].prototype), data.handler);
        }
        this.completionChecker = data.completionChecker;
    }
    matches(request) {
        return matchers.matchesAll(request, this.matchers);
    }
    handle(req, res, head, record) {
        let handlerPromise = (async () => {
            return this.handler.handle(req, res, head);
        })();
        // Requests are added to rule.requests as soon as they start being handled,
        // as promises, which resolve only when the response & request body is complete.
        if (record) {
            this.requests.push(Promise.race([
                // When the handler resolves, the request is completed:
                handlerPromise,
                // If the response is closed before the handler completes (due to aborts, handler
                // timeouts, whatever) then that also counts as the request being completed:
                new Promise((resolve) => res.on('close', resolve))
            ])
                .catch(() => { }) // Ignore handler errors here - we're only tracking the request
                .then(() => (0, request_utils_1.waitForCompletedRequest)(req)));
        }
        // Even if traffic recording is disabled, the number of matched
        // requests is still tracked
        this.requestCount += 1;
        return handlerPromise;
    }
    isComplete() {
        if (this.completionChecker) {
            // If we have a specific rule, use that
            return this.completionChecker.isComplete(this.requestCount);
        }
        else if (this.requestCount === 0) {
            // Otherwise, by default we're definitely incomplete if we've seen no requests
            return false;
        }
        else {
            // And we're _maybe_ complete if we've seen at least one request. In reality, we're incomplete
            // but we should be used anyway if we're at any point we're the last matching rule for a request.
            return null;
        }
    }
    explain(withoutExactCompletion = false) {
        let explanation = `Match websockets ${matchers.explainMatchers(this.matchers)}, ` +
            `and then ${this.handler.explain()}`;
        if (this.completionChecker) {
            explanation += `, ${this.completionChecker.explain(withoutExactCompletion ? undefined : this.requestCount)}.`;
        }
        else {
            explanation += '.';
        }
        return explanation;
    }
    dispose() {
        this.handler.dispose();
        this.matchers.forEach(m => m.dispose());
        if (this.completionChecker)
            this.completionChecker.dispose();
    }
}
exports.WebSocketRule = WebSocketRule;
//# sourceMappingURL=websocket-rule.js.map