"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateHeader = exports.dropDefaultHeaders = exports.h1HeadersToH2 = exports.h2HeadersToH1 = exports.objectHeadersToFlat = exports.objectHeadersToRaw = exports.rawHeadersToObjectPreservingCase = exports.rawHeadersToObject = exports.flattenPairedRawHeaders = exports.pairFlatRawHeaders = exports.findRawHeaders = exports.findRawHeaderIndex = exports.findRawHeader = void 0;
const http = require("http");
/*

These utils support conversion between the various header representations that we deal
with. Those are:

- Flat arrays of [key, value, key, value, key, ...]. This is the raw header format
  generally used by Node.js's APIs throughout.
- Raw header tuple arrays like [[key, value], [key, value]]. This is our own raw header
  format, aiming to be fairly easy to use and to preserve header order, header dupes &
  header casing throughout.
- Formatted header objects of { key: value, key: value }. These are returned as the most
  convenient and consistent header format: keys are lowercased, and values are either
  strings or arrays of strings (for duplicate headers). This is returned by Node's APIs,
  but with some unclear normalization rules, so in practice we build raw headers and
  reconstruct this ourselves everyhere, by lowercasing & building arrays of values.

*/
const findRawHeader = (rawHeaders, targetKey) => rawHeaders.find(([key]) => key.toLowerCase() === targetKey);
exports.findRawHeader = findRawHeader;
const findRawHeaderIndex = (rawHeaders, targetKey) => rawHeaders.findIndex(([key]) => key.toLowerCase() === targetKey);
exports.findRawHeaderIndex = findRawHeaderIndex;
const findRawHeaders = (rawHeaders, targetKey) => rawHeaders.filter(([key]) => key.toLowerCase() === targetKey);
exports.findRawHeaders = findRawHeaders;
/**
 * Return node's _very_ raw headers ([k, v, k, v, ...]) into our slightly more convenient
 * pairwise tuples [[k, v], [k, v], ...] RawHeaders structure.
 */
function pairFlatRawHeaders(flatRawHeaders) {
    const result = [];
    for (let i = 0; i < flatRawHeaders.length; i += 2 /* Move two at a time */) {
        result[i / 2] = [flatRawHeaders[i], flatRawHeaders[i + 1]];
    }
    return result;
}
exports.pairFlatRawHeaders = pairFlatRawHeaders;
function flattenPairedRawHeaders(rawHeaders) {
    return rawHeaders.flat();
}
exports.flattenPairedRawHeaders = flattenPairedRawHeaders;
/**
 * Take a raw headers, and turn them into headers, but without some of Node's concessions
 * to ease of use, i.e. keeping multiple values as arrays.
 *
 * This lowercases all names along the way, to provide a convenient header API for most
 * downstream use cases, and to match Node's own behaviour.
 */
function rawHeadersToObject(rawHeaders) {
    return rawHeaders.reduce((headers, [key, value]) => {
        key = key.toLowerCase();
        const existingValue = headers[key];
        if (Array.isArray(existingValue)) {
            existingValue.push(value);
        }
        else if (existingValue) {
            headers[key] = [existingValue, value];
        }
        else {
            headers[key] = value;
        }
        return headers;
    }, {});
}
exports.rawHeadersToObject = rawHeadersToObject;
/**
 * Take raw headers, and turn them into headers just like `rawHeadersToObject` but
 * also preserves case en route.
 *
 * This is separated because our public APIs should _not_ do this, but there's a few
 * internal use cases where we want to, notably including passing headers to WS which
 * only accepts a headers object when sending upstream requests, but does preserve
 * case from the object.
 */
function rawHeadersToObjectPreservingCase(rawHeaders) {
    // Duplicate keys with different cases in the final object clobber each other (last
    // value wins) so we need to pick a single casing for each header name. We don't want
    // to just use lowercase, because we want to preserve original casing wherever possible.
    // To make that work, we use the casing from the first instance of each header, along with
    // a lowercase -> first casing map here to look up that value later:
    const headerNameMap = {};
    return rawHeaders.reduce((headers, [key, value]) => {
        const lowerCaseKey = key.toLowerCase();
        if (headerNameMap[lowerCaseKey]) {
            // If we've already seen this header, we need to use the same
            // casing as before to avoid issues with duplicates:
            key = headerNameMap[lowerCaseKey];
        }
        else {
            // If we haven't, we store this key as the canonical format
            // to make it easy to merge with any duplicates:
            headerNameMap[lowerCaseKey] = key;
        }
        const existingValue = headers[key];
        if (Array.isArray(existingValue)) {
            existingValue.push(value);
        }
        else if (existingValue) {
            headers[key] = [existingValue, value];
        }
        else {
            headers[key] = value;
        }
        return headers;
    }, {});
}
exports.rawHeadersToObjectPreservingCase = rawHeadersToObjectPreservingCase;
function objectHeadersToRaw(headers) {
    const rawHeaders = [];
    for (let key in headers) {
        const value = headers[key];
        if (value === undefined)
            continue; // Drop undefined header values
        if (Array.isArray(value)) {
            value.forEach((v) => rawHeaders.push([key, v]));
        }
        else {
            rawHeaders.push([key, value]);
        }
    }
    return rawHeaders;
}
exports.objectHeadersToRaw = objectHeadersToRaw;
function objectHeadersToFlat(headers) {
    const flatHeaders = [];
    for (let key in headers) {
        const value = headers[key];
        if (value === undefined)
            continue; // Drop undefined header values
        if (Array.isArray(value)) {
            value.forEach((v) => {
                flatHeaders.push(key);
                flatHeaders.push(v);
            });
        }
        else {
            flatHeaders.push(key);
            flatHeaders.push(value);
        }
    }
    return flatHeaders;
}
exports.objectHeadersToFlat = objectHeadersToFlat;
// See https://httptoolkit.tech/blog/translating-http-2-into-http-1/ for details on the
// transformations required between H2 & H1 when proxying.
function h2HeadersToH1(h2Headers) {
    let h1Headers = h2Headers.filter(([key]) => key[0] !== ':');
    if (!(0, exports.findRawHeader)(h1Headers, 'host') && (0, exports.findRawHeader)(h2Headers, ':authority')) {
        h1Headers.unshift(['Host', (0, exports.findRawHeader)(h2Headers, ':authority')[1]]);
    }
    // In HTTP/1 you MUST only send one cookie header - in HTTP/2 sending multiple is fine,
    // so we have to concatenate them:
    const cookieHeaders = (0, exports.findRawHeaders)(h1Headers, 'cookie');
    if (cookieHeaders.length > 1) {
        h1Headers = h1Headers.filter(([key]) => key.toLowerCase() !== 'cookie');
        h1Headers.push(['Cookie', cookieHeaders.join('; ')]);
    }
    return h1Headers;
}
exports.h2HeadersToH1 = h2HeadersToH1;
// Take from http2/util.js in Node itself
const HTTP2_ILLEGAL_HEADERS = [
    'connection',
    'upgrade',
    'host',
    'http2-settings',
    'keep-alive',
    'proxy-connection',
    'transfer-encoding'
];
function h1HeadersToH2(headers) {
    return headers.filter(([key]) => !HTTP2_ILLEGAL_HEADERS.includes(key.toLowerCase()));
}
exports.h1HeadersToH2 = h1HeadersToH2;
// If the user explicitly specifies headers, we tell Node not to handle them,
// so the user-defined headers are the full set.
function dropDefaultHeaders(response) {
    // Drop the default headers, so only the headers we explicitly configure are included
    [
        'connection',
        'content-length',
        'transfer-encoding',
        'date'
    ].forEach((defaultHeader) => response.removeHeader(defaultHeader));
}
exports.dropDefaultHeaders = dropDefaultHeaders;
function validateHeader(name, value) {
    try {
        http.validateHeaderName(name);
        http.validateHeaderValue(name, value);
        return true;
    }
    catch (e) {
        return false;
    }
}
exports.validateHeader = validateHeader;
//# sourceMappingURL=header-utils.js.map